﻿using NVCC.Models;
using System.Collections.Generic;

namespace NVCC.WebUI.Infrastructure
{
    public class UserService : IUserService
    {
        private readonly IUserRepository _userRepo;
        private readonly IUserSessionManager _userSessionManager;
        private readonly IViaRepository _viaRepository;
        private readonly ICurrentUser _currentUser;
      
        public UserService(IUserRepository userRepo, 
                           IUserSessionManager userSessionManager, 
                           IViaRepository viaRepository,
                           ICurrentUser currentUser)
        {
            _userRepo = userRepo;
            _userSessionManager = userSessionManager;
            _viaRepository = viaRepository;
            _currentUser = currentUser;
        }

        public User GetUser()
        {
            var userProfile = _userSessionManager.UserProfile;
            if (userProfile != null) return userProfile;
            // not cached in session manager
            userProfile = _userRepo.GetUser(_currentUser.Name); 
            _userSessionManager.UserProfile = userProfile;
            return userProfile;
        }

        // Updates in persistent storage the record for the current default station
        public void UpdateUserFacility(User user)
        {
            _userRepo.UpdateUserFacility(user.ApplicationUserID, user.Facilities[user.CurrentDefaultFacility]);
        }

        public ViaUser GetViaUser(string userName, string password, string station, short sta3n)
        {
            ViaUser viaUser = _viaRepository.Login(station, userName, password);
            if (HttpContextManager.Current.Session != null) 
                HttpContextManager.Current.Session["ViaUser-" + sta3n.ToString()] = viaUser;

            return viaUser;
        }

        public ViaUser GetViaUser(short sta3n)
        {
            ViaUser viaUser = null;
            if (HttpContextManager.Current.Server != null)
            {
                viaUser = HttpContextManager.Current.Session["ViaUser-" + sta3n.ToString()] as ViaUser;
            }
            return viaUser; // returns null if not found
        }
      
        public StationInfo GetStationInfo(short station)
        {
            var stationInfo = _userRepo.GetStationInfo(station);
            return stationInfo;
        }

        public void SaveStationInfo(StationInfo stationInfo)
        {
            _userRepo.SaveStationInfo(stationInfo);
        }

        public IEnumerable<Division> GetDivisions(int station)
        {
           var divisions= _userRepo.GetDivisions(station);
            return divisions;
        }
        public Division GetDivision(string sta6a)
        {
            var division = _userRepo.GetDivision(sta6a);
            return division;
        }
        public IList<Visn> GetVisn()
        {
            var visn = _userRepo.GetVisn();
            return visn;
        }

        public IList<Facility> GetFacilityByVisnId()
        {
            var facilityByVisnId = _userRepo.GetFacilityByVisnId();
            return facilityByVisnId;
        }

        public int CheckRoiAdminAuthorization()
        {
            return _userRepo.CheckRoiAdminAuthorization(HttpContextManager.Current.User.Identity.Name);
        }
    }
}